/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessrolechanges;

import java.time.ZonedDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessrolechanges.selectable.BusinessRoleChangesSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessrolechanges.BusinessRoleChanges
 * BusinessRoleChanges} entity using key fields. This fluent helper allows methods which modify the underlying query to
 * be called before executing the query itself.
 *
 */
public class BusinessRoleChangesByKeyFluentHelper
    extends
    FluentHelperByKey<BusinessRoleChangesByKeyFluentHelper, BusinessRoleChanges, BusinessRoleChangesSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessrolechanges.BusinessRoleChanges
     * BusinessRoleChanges} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code BusinessRoleChanges}
     * @param businessRoleID
     *            Business Role ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     * @param changeCategory
     *            Change Category
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code BusinessRoleChanges}
     * @param attribute
     *            Attribute
     *            <p>
     *            Constraints: Not nullable, Maximum length: 256
     *            </p>
     * @param valueChangedFrom
     *            Value Changed From
     *            <p>
     *            Constraints: Not nullable, Maximum length: 254
     *            </p>
     * @param changedOn
     *            Changed On
     *            <p>
     *            Constraints: Not nullable, Precision: 7
     *            </p>
     * @param valueChangedTo
     *            Value Changed To
     *            <p>
     *            Constraints: Not nullable, Maximum length: 254
     *            </p>
     */
    public BusinessRoleChangesByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String businessRoleID,
        final ZonedDateTime changedOn,
        final String changeCategory,
        final String attribute,
        final String valueChangedFrom,
        final String valueChangedTo )
    {
        super(servicePath, entityCollection);
        this.key.put("BusinessRoleID", businessRoleID);
        this.key.put("ChangedOn", changedOn);
        this.key.put("ChangeCategory", changeCategory);
        this.key.put("Attribute", attribute);
        this.key.put("ValueChangedFrom", valueChangedFrom);
        this.key.put("ValueChangedTo", valueChangedTo);
    }

    @Override
    @Nonnull
    protected Class<BusinessRoleChanges> getEntityClass()
    {
        return BusinessRoleChanges.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
