/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesssituation;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>SendActionStatus</b> OData function import.
 *
 */
public class SendActionStatusFluentHelper
    extends
    SingleValuedFluentHelperFunction<SendActionStatusFluentHelper, SendActionStatusReturn, SendActionStatusReturn>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>SendActionStatus</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param sitnActionStatus
     *            Action status: SUCCESS or FAILURE
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 7
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SitnActionStatus</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param sitnActionInstanceID
     *            Unique action instance ID generated by the service client
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 36
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SitnActionInstanceID</b>
     *            </p>
     * @param sitnInstanceID
     *            Situation instance ID for which the action was performed
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 32
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SitnInstanceID</b>
     *            </p>
     * @param sitnActionID
     *            ActionID defined by situation template, NO_ACTION or UNKNOWN
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 32
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SitnActionID</b>
     *            </p>
     * @param sitnInstceTargetStatus
     *            Target instance status: OP(Open) RE(Resolved) UN(Unchanged)
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SitnInstceTargetStatus</b>
     *            </p>
     */
    public SendActionStatusFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String sitnActionInstanceID,
        @Nonnull final String sitnActionID,
        @Nonnull final String sitnActionStatus,
        @Nonnull final String sitnInstanceID,
        @Nonnull final String sitnInstceTargetStatus )
    {
        super(servicePath);
        values.put("SitnActionInstanceID", sitnActionInstanceID);
        values.put("SitnActionID", sitnActionID);
        values.put("SitnActionStatus", sitnActionStatus);
        values.put("SitnInstanceID", sitnInstanceID);
        values.put("SitnInstceTargetStatus", sitnInstceTargetStatus);
    }

    @Override
    @Nonnull
    protected Class<SendActionStatusReturn> getEntityClass()
    {
        return SendActionStatusReturn.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "SendActionStatus";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public SendActionStatusReturn executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
