/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesssolutionorder;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>LinkProjectItem</b> OData function import.
 *
 */
public class LinkProjectItemFluentHelper
    extends
    SingleValuedFluentHelperFunction<LinkProjectItemFluentHelper, Result, Result>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>LinkProjectItem</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param businessSolutionOrder
     *            Object ID
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BusinessSolutionOrder</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param linkedProject
     *            Project
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 24
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>LinkedProject</b>
     *            </p>
     * @param businessSolutionOrderItem
     *            Item Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BusinessSolutionOrderItem</b>
     *            </p>
     */
    public LinkProjectItemFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String linkedProject,
        @Nonnull final String businessSolutionOrderItem,
        @Nonnull final String businessSolutionOrder )
    {
        super(servicePath);
        values.put("LinkedProject", linkedProject);
        values.put("BusinessSolutionOrderItem", businessSolutionOrderItem);
        values.put("BusinessSolutionOrder", businessSolutionOrder);
    }

    @Override
    @Nonnull
    protected Class<Result> getEntityClass()
    {
        return Result.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "LinkProjectItem";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public Result executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
