/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessuserchange;

import java.time.ZonedDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessuserchange.selectable.BusinessUserChangesSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessuserchange.BusinessUserChanges
 * BusinessUserChanges} entity using key fields. This fluent helper allows methods which modify the underlying query to
 * be called before executing the query itself.
 *
 */
public class BusinessUserChangesByKeyFluentHelper
    extends
    FluentHelperByKey<BusinessUserChangesByKeyFluentHelper, BusinessUserChanges, BusinessUserChangesSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessuserchange.BusinessUserChanges
     * BusinessUserChanges} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param businessUserID
     *            Business User ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code BusinessUserChanges}
     * @param changeCategory
     *            Change Category
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code BusinessUserChanges}
     * @param attribute
     *            Attribute
     *            <p>
     *            Constraints: Not nullable, Maximum length: 20
     *            </p>
     * @param valueChangedFrom
     *            Value Changed From
     *            <p>
     *            Constraints: Not nullable, Maximum length: 254
     *            </p>
     * @param changedOn
     *            Changed On
     *            <p>
     *            Constraints: Not nullable, Precision: 7
     *            </p>
     * @param valueChangedTo
     *            Value Changed To
     *            <p>
     *            Constraints: Not nullable, Maximum length: 254
     *            </p>
     */
    public BusinessUserChangesByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String businessUserID,
        final ZonedDateTime changedOn,
        final String changeCategory,
        final String attribute,
        final String valueChangedFrom,
        final String valueChangedTo )
    {
        super(servicePath, entityCollection);
        this.key.put("BusinessUserID", businessUserID);
        this.key.put("ChangedOn", changedOn);
        this.key.put("ChangeCategory", changeCategory);
        this.key.put("Attribute", attribute);
        this.key.put("ValueChangedFrom", valueChangedFrom);
        this.key.put("ValueChangedTo", valueChangedTo);
    }

    @Override
    @Nonnull
    protected Class<BusinessUserChanges> getEntityClass()
    {
        return BusinessUserChanges.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
