/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.changemasterv2;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ShiftValidityStartDateOfChangeMaster</b> OData function import.
 *
 */
public class ShiftValidityStartDateOfChangeMasterFluentHelper
    extends
    SingleValuedFluentHelperFunction<ShiftValidityStartDateOfChangeMasterFluentHelper, ChangeMaster, ChangeMaster>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ShiftValidityStartDateOfChangeMaster</b> OData function
     * import with the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param validityStartDate
     *            Valid From
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ValidityStartDate</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param changeNumber
     *            Change Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ChangeNumber</b>
     *            </p>
     */
    public ShiftValidityStartDateOfChangeMasterFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String changeNumber,
        @Nullable final LocalDateTime validityStartDate )
    {
        super(servicePath);
        values.put("ChangeNumber", changeNumber);
        if( validityStartDate != null ) {
            values.put("ValidityStartDate", validityStartDate);
        }
    }

    @Override
    @Nonnull
    protected Class<ChangeMaster> getEntityClass()
    {
        return ChangeMaster.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ShiftValidityStartDateOfChangeMaster";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public ChangeMaster executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
