/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.selectable.DemandSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.Demand
 * Demand} entity using key fields. This fluent helper allows methods which modify the underlying query to be called
 * before executing the query itself.
 *
 */
public class DemandByKeyFluentHelper extends FluentHelperByKey<DemandByKeyFluentHelper, Demand, DemandSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.Demand Demand} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code Demand}
     * @param delvryServOrg
     *            Delivery Org. ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 5
     *            </p>
     * @param resourceId
     *            Resource ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     * @param workPackageID
     *            Work Package ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 50
     *            </p>
     * @param workPackageName
     *            Work Package Name
     *            <p>
     *            Constraints: Not nullable, Maximum length: 60
     *            </p>
     * @param resType
     *            Resource Type ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code Demand}
     * @param workitem
     *            Work Item ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param projectID
     *            Project ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     */
    public DemandByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String projectID,
        final String workPackageID,
        final String workPackageName,
        final String resType,
        final String resourceId,
        final String workitem,
        final String delvryServOrg )
    {
        super(servicePath, entityCollection);
        this.key.put("ProjectID", projectID);
        this.key.put("WorkPackageID", workPackageID);
        this.key.put("WorkPackageName", workPackageName);
        this.key.put("ResType", resType);
        this.key.put("ResourceId", resourceId);
        this.key.put("Workitem", workitem);
        this.key.put("DelvryServOrg", delvryServOrg);
    }

    @Override
    @Nonnull
    protected Class<Demand> getEntityClass()
    {
        return Demand.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
