/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.selectable.ProjectRoleSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.ProjectRole ProjectRole} entity using
 * key fields. This fluent helper allows methods which modify the underlying query to be called before executing the
 * query itself.
 *
 */
public class ProjectRoleByKeyFluentHelper
    extends
    FluentHelperByKey<ProjectRoleByKeyFluentHelper, ProjectRole, ProjectRoleSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.ProjectRole ProjectRole} entity with
     * the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on
     * the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ProjectRole}
     * @param servicePath
     *            Service path to be used to fetch a single {@code ProjectRole}
     * @param projectID
     *            Project ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     * @param projectRoleID
     *            Project Role ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 15
     *            </p>
     */
    public ProjectRoleByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String projectID,
        final String projectRoleID )
    {
        super(servicePath, entityCollection);
        this.key.put("ProjectID", projectID);
        this.key.put("ProjectRoleID", projectRoleID);
    }

    @Override
    @Nonnull
    protected Class<ProjectRole> getEntityClass()
    {
        return ProjectRole.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
