/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.selectable.WorkPackageSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.WorkPackage WorkPackage} entity using
 * key fields. This fluent helper allows methods which modify the underlying query to be called before executing the
 * query itself.
 *
 */
public class WorkPackageByKeyFluentHelper
    extends
    FluentHelperByKey<WorkPackageByKeyFluentHelper, WorkPackage, WorkPackageSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.commercialproject.WorkPackage WorkPackage} entity with
     * the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on
     * the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code WorkPackage}
     * @param workPackageID
     *            Work Package ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 50
     *            </p>
     * @param workPackageName
     *            Work Package Name
     *            <p>
     *            Constraints: Not nullable, Maximum length: 60
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code WorkPackage}
     * @param projectID
     *            Project ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     */
    public WorkPackageByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String projectID,
        final String workPackageID,
        final String workPackageName )
    {
        super(servicePath, entityCollection);
        this.key.put("ProjectID", projectID);
        this.key.put("WorkPackageID", workPackageID);
        this.key.put("WorkPackageName", workPackageName);
    }

    @Override
    @Nonnull
    protected Class<WorkPackage> getEntityClass()
    {
        return WorkPackage.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
