/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcentertest;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcentertest.selectable.CostCenterSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcentertest.CostCenter
 * CostCenter} entity using key fields. This fluent helper allows methods which modify the underlying query to be called
 * before executing the query itself.
 *
 */
public class CostCenterByKeyFluentHelper
    extends
    FluentHelperByKey<CostCenterByKeyFluentHelper, CostCenter, CostCenterSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcentertest.CostCenter CostCenter} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param companyCode
     *            The company code is an organizational unit within financial accounting.
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code CostCenter}
     * @param costCenterID
     *            A key that uniquely identifies a cost center.
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code CostCenter}
     * @param validityEndDate
     *            Date indicating up to when an entry is valid.
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     */
    public CostCenterByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String companyCode,
        final String costCenterID,
        final LocalDateTime validityEndDate )
    {
        super(servicePath, entityCollection);
        this.key.put("CompanyCode", companyCode);
        this.key.put("CostCenterID", costCenterID);
        this.key.put("ValidityEndDate", validityEndDate);
    }

    @Override
    @Nonnull
    protected Class<CostCenter> getEntityClass()
    {
        return CostCenter.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
