/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcentertest;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcentertest.selectable.CostCenterTextSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcentertest.CostCenterText CostCenterText} entity using
 * key fields. This fluent helper allows methods which modify the underlying query to be called before executing the
 * query itself.
 *
 */
public class CostCenterTextByKeyFluentHelper
    extends
    FluentHelperByKey<CostCenterTextByKeyFluentHelper, CostCenterText, CostCenterTextSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcentertest.CostCenterText CostCenterText} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param companyCode
     *            The company code is an organizational unit within financial accounting.
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code CostCenterText}
     * @param costCenterID
     *            A key that uniquely identifies a cost center.
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code CostCenterText}
     * @param language
     *            SAP Language Key (2-letter code) for Unicode languages.
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param validityEndDate
     *            Date indicating up to when an entry is valid.
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     */
    public CostCenterTextByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String companyCode,
        final String costCenterID,
        final LocalDateTime validityEndDate,
        final String language )
    {
        super(servicePath, entityCollection);
        this.key.put("CompanyCode", companyCode);
        this.key.put("CostCenterID", costCenterID);
        this.key.put("ValidityEndDate", validityEndDate);
        this.key.put("Language", language);
    }

    @Override
    @Nonnull
    protected Class<CostCenterText> getEntityClass()
    {
        return CostCenterText.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
