/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.costcontrollingjournalentry;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>Reverse</b> OData function import.
 *
 */
public class ReverseFluentHelper
    extends
    SingleValuedFluentHelperFunction<ReverseFluentHelper, CostRevenueReassignment, CostRevenueReassignment>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>Reverse</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param documentDate
     *            Document Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DocumentDate</b>
     *            </p>
     * @param accountingDocumentHeaderText
     *            Document Header Text
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 25
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>AccountingDocumentHeaderText</b>
     *            </p>
     * @param warningsAreIgnored
     *            Ignore warning
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>WarningsAreIgnored</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param referenceDocument
     *            Reference document
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ReferenceDocument</b>
     *            </p>
     * @param fiscalPeriod
     *            Fiscal Period
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FiscalPeriod</b>
     *            </p>
     * @param controllingArea
     *            Controlling Area
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ControllingArea</b>
     *            </p>
     * @param postingDate
     *            Posting Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PostingDate</b>
     *            </p>
     * @param fiscalYear
     *            Fiscal Year
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FiscalYear</b>
     *            </p>
     */
    public ReverseFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String fiscalYear,
        @Nonnull final String referenceDocument,
        @Nonnull final String controllingArea,
        @Nullable final LocalDateTime documentDate,
        @Nullable final LocalDateTime postingDate,
        @Nullable final String fiscalPeriod,
        @Nullable final String accountingDocumentHeaderText,
        @Nullable final String warningsAreIgnored )
    {
        super(servicePath);
        values.put("FiscalYear", fiscalYear);
        values.put("ReferenceDocument", referenceDocument);
        values.put("ControllingArea", controllingArea);
        if( documentDate != null ) {
            values.put("DocumentDate", documentDate);
        }
        if( postingDate != null ) {
            values.put("PostingDate", postingDate);
        }
        if( fiscalPeriod != null ) {
            values.put("FiscalPeriod", fiscalPeriod);
        }
        if( accountingDocumentHeaderText != null ) {
            values.put("AccountingDocumentHeaderText", accountingDocumentHeaderText);
        }
        if( warningsAreIgnored != null ) {
            values.put("WarningsAreIgnored", warningsAreIgnored);
        }
    }

    @Override
    @Nonnull
    protected Class<CostRevenueReassignment> getEntityClass()
    {
        return CostRevenueReassignment.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "Reverse";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public CostRevenueReassignment executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
