/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.customermaterial;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.customermaterial.selectable.CustomerMaterialSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.customermaterial.CustomerMaterial CustomerMaterial} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class CustomerMaterialByKeyFluentHelper
    extends
    FluentHelperByKey<CustomerMaterialByKeyFluentHelper, CustomerMaterial, CustomerMaterialSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.customermaterial.CustomerMaterial CustomerMaterial}
     * entity with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest}
     * method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code CustomerMaterial}
     * @param material
     *            Product Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code CustomerMaterial}
     * @param distributionChannel
     *            Distribution Channel
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param salesOrganization
     *            Sales Organization
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param customer
     *            Customer number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public CustomerMaterialByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String salesOrganization,
        final String distributionChannel,
        final String customer,
        final String material )
    {
        super(servicePath, entityCollection);
        this.key.put("SalesOrganization", salesOrganization);
        this.key.put("DistributionChannel", distributionChannel);
        this.key.put("Customer", customer);
        this.key.put("Material", material);
    }

    @Override
    @Nonnull
    protected Class<CustomerMaterial> getEntityClass()
    {
        return CustomerMaterial.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
