/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.defectprocessing;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.defectprocessing.selectable.DefectSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.defectprocessing.Defect
 * Defect} entity using key fields. This fluent helper allows methods which modify the underlying query to be called
 * before executing the query itself.
 *
 */
public class DefectByKeyFluentHelper extends FluentHelperByKey<DefectByKeyFluentHelper, Defect, DefectSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.defectprocessing.Defect Defect} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code Defect}
     * @param defectInternalID
     *            Notification Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code Defect}
     */
    public DefectByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String defectInternalID )
    {
        super(servicePath, entityCollection);
        this.key.put("DefectInternalID", defectInternalID);
    }

    @Override
    @Nonnull
    protected Class<Defect> getEntityClass()
    {
        return Defect.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
