/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.documentmanagement;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.documentmanagement.selectable.DocumentInfoRecordClassSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.documentmanagement.DocumentInfoRecordClass
 * DocumentInfoRecordClass} entity using key fields. This fluent helper allows methods which modify the underlying query
 * to be called before executing the query itself.
 *
 */
public class DocumentInfoRecordClassByKeyFluentHelper
    extends
    FluentHelperByKey<DocumentInfoRecordClassByKeyFluentHelper, DocumentInfoRecordClass, DocumentInfoRecordClassSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.documentmanagement.DocumentInfoRecordClass
     * DocumentInfoRecordClass} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code DocumentInfoRecordClass}
     * @param documentInfoRecordDocType
     *            Document Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param documentInfoRecordDocNumber
     *            Document number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 25
     *            </p>
     * @param classInternalID
     *            Internal Class Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param documentInfoRecordDocPart
     *            Document Part
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code DocumentInfoRecordClass}
     * @param documentInfoRecordDocVersion
     *            Document Version
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     */
    public DocumentInfoRecordClassByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String documentInfoRecordDocType,
        final String documentInfoRecordDocNumber,
        final String documentInfoRecordDocVersion,
        final String documentInfoRecordDocPart,
        final String classInternalID )
    {
        super(servicePath, entityCollection);
        this.key.put("DocumentInfoRecordDocType", documentInfoRecordDocType);
        this.key.put("DocumentInfoRecordDocNumber", documentInfoRecordDocNumber);
        this.key.put("DocumentInfoRecordDocVersion", documentInfoRecordDocVersion);
        this.key.put("DocumentInfoRecordDocPart", documentInfoRecordDocPart);
        this.key.put("ClassInternalID", classInternalID);
    }

    @Override
    @Nonnull
    protected Class<DocumentInfoRecordClass> getEntityClass()
    {
        return DocumentInfoRecordClass.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
