/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment;

import java.net.URI;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>DismantleEquipment</b> OData function import.
 *
 */
public class DismantleEquipmentFluentHelper
    extends
    SingleValuedFluentHelperFunction<DismantleEquipmentFluentHelper, Equipment, Equipment>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>DismantleEquipment</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param superordinateEquipment
     *            Superord. Equipment
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 18
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SuperordinateEquipment</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param functionalLocation
     *            Functional Location
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FunctionalLocation</b>
     *            </p>
     * @param equipmentInstallationTime
     *            Time
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>EquipmentInstallationTime</b>
     *            </p>
     * @param equipment
     *            Equipment
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 18
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Equipment</b>
     *            </p>
     * @param equipInstallationPositionNmbr
     *            Item (SD)
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 6
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>EquipInstallationPositionNmbr</b>
     *            </p>
     * @param validityEndDate
     *            Valid To
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ValidityEndDate</b>
     *            </p>
     * @param equipmentInstallationDate
     *            Changed On
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>EquipmentInstallationDate</b>
     *            </p>
     */
    public DismantleEquipmentFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String equipment,
        @Nonnull final LocalDateTime validityEndDate,
        @Nullable final String superordinateEquipment,
        @Nullable final String equipInstallationPositionNmbr,
        @Nullable final String functionalLocation,
        @Nullable final LocalDateTime equipmentInstallationDate,
        @Nullable final LocalTime equipmentInstallationTime )
    {
        super(servicePath);
        values.put("Equipment", equipment);
        values.put("ValidityEndDate", validityEndDate);
        if( superordinateEquipment != null ) {
            values.put("SuperordinateEquipment", superordinateEquipment);
        }
        if( equipInstallationPositionNmbr != null ) {
            values.put("EquipInstallationPositionNmbr", equipInstallationPositionNmbr);
        }
        if( functionalLocation != null ) {
            values.put("FunctionalLocation", functionalLocation);
        }
        if( equipmentInstallationDate != null ) {
            values.put("EquipmentInstallationDate", equipmentInstallationDate);
        }
        if( equipmentInstallationTime != null ) {
            values.put("EquipmentInstallationTime", equipmentInstallationTime);
        }
    }

    @Override
    @Nonnull
    protected Class<Equipment> getEntityClass()
    {
        return Equipment.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "DismantleEquipment";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public Equipment executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
