/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment.selectable.EquipmentSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment.Equipment
 * Equipment} entity using key fields. This fluent helper allows methods which modify the underlying query to be called
 * before executing the query itself.
 *
 */
public class EquipmentByKeyFluentHelper
    extends
    FluentHelperByKey<EquipmentByKeyFluentHelper, Equipment, EquipmentSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment.Equipment Equipment} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code Equipment}
     * @param servicePath
     *            Service path to be used to fetch a single {@code Equipment}
     * @param equipment
     *            Equipment Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 18
     *            </p>
     * @param validityEndDate
     *            Valid To Date
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     */
    public EquipmentByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String equipment,
        final LocalDateTime validityEndDate )
    {
        super(servicePath, entityCollection);
        this.key.put("Equipment", equipment);
        this.key.put("ValidityEndDate", validityEndDate);
    }

    @Override
    @Nonnull
    protected Class<Equipment> getEntityClass()
    {
        return Equipment.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
