/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment.selectable.EquipmentWarrantySelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment.EquipmentWarranty EquipmentWarranty} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class EquipmentWarrantyByKeyFluentHelper
    extends
    FluentHelperByKey<EquipmentWarrantyByKeyFluentHelper, EquipmentWarranty, EquipmentWarrantySelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.equipment.EquipmentWarranty EquipmentWarranty} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code EquipmentWarranty}
     * @param servicePath
     *            Service path to be used to fetch a single {@code EquipmentWarranty}
     * @param equipment
     *            Equipment Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 18
     *            </p>
     * @param warrantyType
     *            Warranty type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     */
    public EquipmentWarrantyByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String equipment,
        final String warrantyType )
    {
        super(servicePath, entityCollection);
        this.key.put("Equipment", equipment);
        this.key.put("WarrantyType", warrantyType);
    }

    @Override
    @Nonnull
    protected Class<EquipmentWarranty> getEntityClass()
    {
        return EquipmentWarranty.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
