/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.eventbasedrevenuerecognitionintegration;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>CallAction</b> OData function import.
 *
 */
public class CallActionFluentHelper
    extends
    SingleValuedFluentHelperFunction<CallActionFluentHelper, Et_ActionResult, Et_ActionResult>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>CallAction</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param period
     *            Signed INT2 / int16
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: none
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Period</b>
     *            </p>
     * @param year
     *            Signed INT2 / int16
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: none
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Year</b>
     *            </p>
     * @param wBSElementExternalID
     *            Constraints: Not nullable, Maximum length: 24
     *            <p>
     *            Original parameter name from the Odata EDM: <b>WBSElementExternalID</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param actionName
     *            Constraints: Not nullable, Maximum length: 1
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ActionName</b>
     *            </p>
     */
    public CallActionFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final Short year,
        @Nonnull final Short period,
        @Nonnull final String actionName,
        @Nonnull final String wBSElementExternalID )
    {
        super(servicePath);
        values.put("Year", year);
        values.put("Period", period);
        values.put("ActionName", actionName);
        values.put("WBSElementExternalID", wBSElementExternalID);
    }

    @Override
    @Nonnull
    protected Class<Et_ActionResult> getEntityClass()
    {
        return Et_ActionResult.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "CallAction";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpGet(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public Et_ActionResult executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
