/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.externaljobschedulerintegration;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>JobCancel</b> OData function import.
 *
 */
public class JobCancelFluentHelper
    extends
    SingleValuedFluentHelperFunction<JobCancelFluentHelper, FuncImpReturn, FuncImpReturn>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>JobCancel</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param jobName
     *            Constraints: Not nullable, Maximum length: 32
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobName</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param jobRunCount
     *            Constraints: Not nullable, Maximum length: 14
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobRunCount</b>
     *            </p>
     */
    public JobCancelFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String jobRunCount,
        @Nonnull final String jobName )
    {
        super(servicePath);
        values.put("JobRunCount", jobRunCount);
        values.put("JobName", jobName);
    }

    @Override
    @Nonnull
    protected Class<FuncImpReturn> getEntityClass()
    {
        return FuncImpReturn.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "JobCancel";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public FuncImpReturn executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
