/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.externaljobschedulerintegration;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>JobSchedule</b> OData function import.
 *
 */
public class JobScheduleFluentHelper
    extends
    SingleValuedFluentHelperFunction<JobScheduleFluentHelper, JobScheduleStatus, JobScheduleStatus>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>JobSchedule</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param testModeInd
     *            Indicator
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>TestModeInd</b>
     *            </p>
     * @param jobText
     *            Constraints: Not nullable, Maximum length: 120
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobText</b>
     *            </p>
     * @param jobUserName
     *            Constraints: Nullable, Maximum length: 12
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobUserName</b>
     *            </p>
     * @param jobParameterValues
     *            Constraints: Nullable
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobParameterValues</b>
     *            </p>
     * @param jobUser
     *            Constraints: Nullable, Maximum length: 40
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobUser</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param jobUserID
     *            Constraints: Nullable, Maximum length: 40
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobUserID</b>
     *            </p>
     * @param jobTemplateName
     *            Constraints: Not nullable, Maximum length: 40
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobTemplateName</b>
     *            </p>
     */
    public JobScheduleFluentHelper(
        @Nonnull final String servicePath,
        @Nullable final String jobUserID,
        @Nullable final String jobUserName,
        @Nonnull final String jobText,
        @Nonnull final String jobTemplateName,
        @Nullable final String jobParameterValues,
        @Nullable final Boolean testModeInd,
        @Nullable final String jobUser )
    {
        super(servicePath);
        if( jobUserID != null ) {
            values.put("JobUserID", jobUserID);
        }
        if( jobUserName != null ) {
            values.put("JobUserName", jobUserName);
        }
        values.put("JobText", jobText);
        values.put("JobTemplateName", jobTemplateName);
        if( jobParameterValues != null ) {
            values.put("JobParameterValues", jobParameterValues);
        }
        if( testModeInd != null ) {
            values.put("TestModeInd", testModeInd);
        }
        if( jobUser != null ) {
            values.put("JobUser", jobUser);
        }
    }

    @Override
    @Nonnull
    protected Class<JobScheduleStatus> getEntityClass()
    {
        return JobScheduleStatus.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "JobSchedule";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public JobScheduleStatus executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
