/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.externaljobschedulerintegration;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>JobStatusGet</b> OData function import.
 *
 */
public class JobStatusGetFluentHelper
    extends
    SingleValuedFluentHelperFunction<JobStatusGetFluentHelper, JobScheduleStatus, JobScheduleStatus>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>JobStatusGet</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param jobName
     *            Constraints: none
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobName</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param jobRunCount
     *            Constraints: none
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JobRunCount</b>
     *            </p>
     */
    public JobStatusGetFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String jobName,
        @Nonnull final String jobRunCount )
    {
        super(servicePath);
        values.put("JobName", jobName);
        values.put("JobRunCount", jobRunCount);
    }

    @Override
    @Nonnull
    protected Class<JobScheduleStatus> getEntityClass()
    {
        return JobScheduleStatus.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "JobStatusGet";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpGet(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public JobScheduleStatus executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
