/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation.selectable.CharacteristicSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation.Characteristic Characteristic} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class CharacteristicByKeyFluentHelper
    extends
    FluentHelperByKey<CharacteristicByKeyFluentHelper, Characteristic, CharacteristicSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation.Characteristic Characteristic}
     * entity with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest}
     * method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code Characteristic}
     * @param charcInternalID
     *            Characteristic Internal ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param classInternalID
     *            Internal Class Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code Characteristic}
     * @param functionalLocation
     *            Key of Object to be Classified
     *            <p>
     *            Constraints: Not nullable, Maximum length: 90
     *            </p>
     */
    public CharacteristicByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String functionalLocation,
        final String classInternalID,
        final String charcInternalID )
    {
        super(servicePath, entityCollection);
        this.key.put("FunctionalLocation", functionalLocation);
        this.key.put("ClassInternalID", classInternalID);
        this.key.put("CharcInternalID", charcInternalID);
    }

    @Override
    @Nonnull
    protected Class<Characteristic> getEntityClass()
    {
        return Characteristic.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
