/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation;

import java.net.URI;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>DismantleFunctionalLocation</b> OData function import.
 *
 */
public class DismantleFunctionalLocationFluentHelper
    extends
    SingleValuedFluentHelperFunction<DismantleFunctionalLocationFluentHelper, FunctionalLocation, FunctionalLocation>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>DismantleFunctionalLocation</b> OData function import
     * with the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param funcLocDismantleTime
     *            Dismantle Time
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FuncLocDismantleTime</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param functionalLocation
     *            Functional location
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 30
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FunctionalLocation</b>
     *            </p>
     * @param funcLocDismantleDate
     *            Dismantle Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FuncLocDismantleDate</b>
     *            </p>
     */
    public DismantleFunctionalLocationFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String functionalLocation,
        @Nullable final LocalDateTime funcLocDismantleDate,
        @Nullable final LocalTime funcLocDismantleTime )
    {
        super(servicePath);
        values.put("FunctionalLocation", functionalLocation);
        if( funcLocDismantleDate != null ) {
            values.put("FuncLocDismantleDate", funcLocDismantleDate);
        }
        if( funcLocDismantleTime != null ) {
            values.put("FuncLocDismantleTime", funcLocDismantleTime);
        }
    }

    @Override
    @Nonnull
    protected Class<FunctionalLocation> getEntityClass()
    {
        return FunctionalLocation.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "DismantleFunctionalLocation";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public FunctionalLocation executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
