/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation.selectable.FunctionalLocationPartnerSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation.FunctionalLocationPartner
 * FunctionalLocationPartner} entity using key fields. This fluent helper allows methods which modify the underlying
 * query to be called before executing the query itself.
 *
 */
public class FunctionalLocationPartnerByKeyFluentHelper
    extends
    FluentHelperByKey<FunctionalLocationPartnerByKeyFluentHelper, FunctionalLocationPartner, FunctionalLocationPartnerSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation.FunctionalLocationPartner
     * FunctionalLocationPartner} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code FunctionalLocationPartner}
     * @param partnerFunction
     *            Partner Function
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code FunctionalLocationPartner}
     * @param functionalLocation
     *            Functional location
     *            <p>
     *            Constraints: Not nullable, Maximum length: 30
     *            </p>
     * @param funcnlLocPartnerObjectNmbr
     *            Counter for differentiation 6-digit
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     */
    public FunctionalLocationPartnerByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String functionalLocation,
        final String partnerFunction,
        final String funcnlLocPartnerObjectNmbr )
    {
        super(servicePath, entityCollection);
        this.key.put("FunctionalLocation", functionalLocation);
        this.key.put("PartnerFunction", partnerFunction);
        this.key.put("FuncnlLocPartnerObjectNmbr", funcnlLocPartnerObjectNmbr);
    }

    @Override
    @Nonnull
    protected Class<FunctionalLocationPartner> getEntityClass()
    {
        return FunctionalLocationPartner.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
