/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.functionallocation;

import java.net.URI;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>InstallFunctionalLocation</b> OData function import.
 *
 */
public class InstallFunctionalLocationFluentHelper
    extends
    SingleValuedFluentHelperFunction<InstallFunctionalLocationFluentHelper, FunctionalLocation, FunctionalLocation>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>InstallFunctionalLocation</b> OData function import with
     * the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param funcLocInstallationTime
     *            Install Time
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FuncLocInstallationTime</b>
     *            </p>
     * @param superiorFunctionalLocation
     *            Superior FunctLoc.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SuperiorFunctionalLocation</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param functionalLocation
     *            Functional location
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 30
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FunctionalLocation</b>
     *            </p>
     * @param funcLocInstallationPosNmbr
     *            Position in object
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FuncLocInstallationPosNmbr</b>
     *            </p>
     * @param funcLocInstallationDate
     *            Install Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FuncLocInstallationDate</b>
     *            </p>
     */
    public InstallFunctionalLocationFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String functionalLocation,
        @Nullable final String superiorFunctionalLocation,
        @Nullable final String funcLocInstallationPosNmbr,
        @Nullable final LocalDateTime funcLocInstallationDate,
        @Nullable final LocalTime funcLocInstallationTime )
    {
        super(servicePath);
        values.put("FunctionalLocation", functionalLocation);
        if( superiorFunctionalLocation != null ) {
            values.put("SuperiorFunctionalLocation", superiorFunctionalLocation);
        }
        if( funcLocInstallationPosNmbr != null ) {
            values.put("FuncLocInstallationPosNmbr", funcLocInstallationPosNmbr);
        }
        if( funcLocInstallationDate != null ) {
            values.put("FuncLocInstallationDate", funcLocInstallationDate);
        }
        if( funcLocInstallationTime != null ) {
            values.put("FuncLocInstallationTime", funcLocInstallationTime);
        }
    }

    @Override
    @Nonnull
    protected Class<FunctionalLocation> getEntityClass()
    {
        return FunctionalLocation.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "InstallFunctionalLocation";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public FunctionalLocation executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
