/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.fund;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.fund.selectable.FundCoreSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.fund.FundCore FundCore}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class FundCoreByKeyFluentHelper
    extends
    FluentHelperByKey<FundCoreByKeyFluentHelper, FundCore, FundCoreSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.fund.FundCore FundCore} entity with the provided key
     * field values. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code FundCore}
     * @param financialManagementArea
     *            Financial Management Area
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param fund
     *            Fund
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code FundCore}
     */
    public FundCoreByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String financialManagementArea,
        final String fund )
    {
        super(servicePath, entityCollection);
        this.key.put("FinancialManagementArea", financialManagementArea);
        this.key.put("Fund", fund);
    }

    @Override
    @Nonnull
    protected Class<FundCore> getEntityClass()
    {
        return FundCore.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
