/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.fund;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.fund.selectable.FundTextSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.fund.FundText FundText}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class FundTextByKeyFluentHelper
    extends
    FluentHelperByKey<FundTextByKeyFluentHelper, FundText, FundTextSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.fund.FundText FundText} entity with the provided key
     * field values. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code FundText}
     * @param financialManagementArea
     *            Financial Management Area
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param fund
     *            Fund
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code FundText}
     * @param language
     *            Language Key
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     */
    public FundTextByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String financialManagementArea,
        final String fund,
        final String language )
    {
        super(servicePath, entityCollection);
        this.key.put("FinancialManagementArea", financialManagementArea);
        this.key.put("Fund", fund);
        this.key.put("Language", language);
    }

    @Override
    @Nonnull
    protected Class<FundText> getEntityClass()
    {
        return FundText.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
