/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.granthierarchy;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.granthierarchy.selectable.GrantCoreHierNodeSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.granthierarchy.GrantCoreHierNode GrantCoreHierNode} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class GrantCoreHierNodeByKeyFluentHelper
    extends
    FluentHelperByKey<GrantCoreHierNodeByKeyFluentHelper, GrantCoreHierNode, GrantCoreHierNodeSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.granthierarchy.GrantCoreHierNode GrantCoreHierNode}
     * entity with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest}
     * method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code GrantCoreHierNode}
     * @param servicePath
     *            Service path to be used to fetch a single {@code GrantCoreHierNode}
     * @param grantHierarchy
     *            Grant Hierarchy
     *            <p>
     *            Constraints: Not nullable, Maximum length: 42
     *            </p>
     * @param validityEndDate
     *            Validity End Date
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     * @param hierarchyNode
     *            Hierarchy node
     *            <p>
     *            Constraints: Not nullable, Maximum length: 50
     *            </p>
     */
    public GrantCoreHierNodeByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String grantHierarchy,
        final String hierarchyNode,
        final LocalDateTime validityEndDate )
    {
        super(servicePath, entityCollection);
        this.key.put("GrantHierarchy", grantHierarchy);
        this.key.put("HierarchyNode", hierarchyNode);
        this.key.put("ValidityEndDate", validityEndDate);
    }

    @Override
    @Nonnull
    protected Class<GrantCoreHierNode> getEntityClass()
    {
        return GrantCoreHierNode.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
