/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.handlingunit;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.handlingunit.selectable.HandlingUnitSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.handlingunit.HandlingUnit
 * HandlingUnit} entity using key fields. This fluent helper allows methods which modify the underlying query to be
 * called before executing the query itself.
 *
 */
public class HandlingUnitByKeyFluentHelper
    extends
    FluentHelperByKey<HandlingUnitByKeyFluentHelper, HandlingUnit, HandlingUnitSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.handlingunit.HandlingUnit HandlingUnit} entity with
     * the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on
     * the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code HandlingUnit}
     * @param handlingUnitExternalID
     *            External Handling Unit Identification
     *            <p>
     *            Constraints: Not nullable, Maximum length: 20
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code HandlingUnit}
     * @param warehouse
     *            Warehouse Number/Warehouse Complex
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public HandlingUnitByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String handlingUnitExternalID,
        final String warehouse )
    {
        super(servicePath, entityCollection);
        this.key.put("HandlingUnitExternalID", handlingUnitExternalID);
        this.key.put("Warehouse", warehouse);
    }

    @Override
    @Nonnull
    protected Class<HandlingUnit> getEntityClass()
    {
        return HandlingUnit.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
