/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.handlingunit;

import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.handlingunit.selectable.HandlingUnitItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.handlingunit.HandlingUnitItem HandlingUnitItem} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class HandlingUnitItemByKeyFluentHelper
    extends
    FluentHelperByKey<HandlingUnitItemByKeyFluentHelper, HandlingUnitItem, HandlingUnitItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.handlingunit.HandlingUnitItem HandlingUnitItem} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code HandlingUnitItem}
     * @param handlingUnitExternalID
     *            External Handling Unit Identification
     *            <p>
     *            Constraints: Not nullable, Maximum length: 20
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code HandlingUnitItem}
     * @param stockItemUUID
     *            GUID Stock Item
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param warehouse
     *            Warehouse Number/Warehouse Complex
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public HandlingUnitItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String handlingUnitExternalID,
        final String warehouse,
        final UUID stockItemUUID )
    {
        super(servicePath, entityCollection);
        this.key.put("HandlingUnitExternalID", handlingUnitExternalID);
        this.key.put("Warehouse", warehouse);
        this.key.put("StockItemUUID", stockItemUUID);
    }

    @Override
    @Nonnull
    protected Class<HandlingUnitItem> getEntityClass()
    {
        return HandlingUnitItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
