/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inbounddeliveryv2;

import java.math.BigDecimal;
import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>CreateBatchSplitItem</b> OData function import.
 *
 */
public class CreateBatchSplitItemFluentHelper
    extends
    SingleValuedFluentHelperFunction<CreateBatchSplitItemFluentHelper, CreatedDeliveryItem, CreatedDeliveryItem>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>CreateBatchSplitItem</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param shelfLifeExpirationDate
     *            Time Stamp
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ShelfLifeExpirationDate</b>
     *            </p>
     * @param deliveryDocument
     *            Constraints: Not nullable, Maximum length: 10
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DeliveryDocument</b>
     *            </p>
     * @param deliveryQuantityUnit
     *            Constraints: Not nullable, Maximum length: 3
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DeliveryQuantityUnit</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param batch
     *            Constraints: Not nullable, Maximum length: 10
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Batch</b>
     *            </p>
     * @param manufactureDate
     *            Time Stamp
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ManufactureDate</b>
     *            </p>
     * @param putawayQuantityInOrderUnit
     *            Constraints: Nullable, Precision: 3, Scale: 3
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PutawayQuantityInOrderUnit</b>
     *            </p>
     * @param actualDeliveryQuantity
     *            Constraints: Not nullable, Precision: 3, Scale: 3
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ActualDeliveryQuantity</b>
     *            </p>
     * @param batchBySupplier
     *            Constraints: Nullable, Maximum length: 15
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BatchBySupplier</b>
     *            </p>
     * @param deliveryDocumentItem
     *            Constraints: Not nullable, Maximum length: 6
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DeliveryDocumentItem</b>
     *            </p>
     */
    public CreateBatchSplitItemFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final BigDecimal actualDeliveryQuantity,
        @Nonnull final String batch,
        @Nonnull final String deliveryDocument,
        @Nonnull final String deliveryDocumentItem,
        @Nonnull final String deliveryQuantityUnit,
        @Nullable final BigDecimal putawayQuantityInOrderUnit,
        @Nullable final LocalDateTime shelfLifeExpirationDate,
        @Nullable final LocalDateTime manufactureDate,
        @Nullable final String batchBySupplier )
    {
        super(servicePath);
        values.put("ActualDeliveryQuantity", actualDeliveryQuantity);
        values.put("Batch", batch);
        values.put("DeliveryDocument", deliveryDocument);
        values.put("DeliveryDocumentItem", deliveryDocumentItem);
        values.put("DeliveryQuantityUnit", deliveryQuantityUnit);
        if( putawayQuantityInOrderUnit != null ) {
            values.put("PutawayQuantityInOrderUnit", putawayQuantityInOrderUnit);
        }
        if( shelfLifeExpirationDate != null ) {
            values.put("ShelfLifeExpirationDate", shelfLifeExpirationDate);
        }
        if( manufactureDate != null ) {
            values.put("ManufactureDate", manufactureDate);
        }
        if( batchBySupplier != null ) {
            values.put("BatchBySupplier", batchBySupplier);
        }
    }

    @Override
    @Nonnull
    protected Class<CreatedDeliveryItem> getEntityClass()
    {
        return CreatedDeliveryItem.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "CreateBatchSplitItem";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public CreatedDeliveryItem executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
