/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inbounddeliveryv2;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.CollectionValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>PostGoodsReceipt</b> OData function import.
 *
 */
public class PostGoodsReceiptFluentHelper
    extends
    CollectionValuedFluentHelperFunction<PostGoodsReceiptFluentHelper, PutawayReport, List<PutawayReport>>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>PostGoodsReceipt</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param deliveryDocument
     *            Delivery
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DeliveryDocument</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     */
    public PostGoodsReceiptFluentHelper( @Nonnull final String servicePath, @Nonnull final String deliveryDocument )
    {
        super(servicePath);
        values.put("DeliveryDocument", deliveryDocument);
    }

    /**
     * Creates a fluent helper object that will execute the <b>PostGoodsReceipt</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param deliveryDocument
     *            Delivery
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DeliveryDocument</b>
     *            </p>
     * @param actualGoodsMovementDate
     *            Time Stamp
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 7
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ActualGoodsMovementDate</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     */
    public PostGoodsReceiptFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String deliveryDocument,
        @Nullable final LocalDateTime actualGoodsMovementDate )
    {
        super(servicePath);
        values.put("DeliveryDocument", deliveryDocument);
        if( actualGoodsMovementDate != null ) {
            values.put("ActualGoodsMovementDate", actualGoodsMovementDate);
        }
    }

    @Override
    @Nonnull
    protected Class<PutawayReport> getEntityClass()
    {
        return PutawayReport.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "PostGoodsReceipt";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nonnull
    public List<PutawayReport> executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeMultiple(destination);
    }

}
