/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inspectionmethod;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inspectionmethod.selectable.InspectionMethodSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inspectionmethod.InspectionMethod InspectionMethod} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class InspectionMethodByKeyFluentHelper
    extends
    FluentHelperByKey<InspectionMethodByKeyFluentHelper, InspectionMethod, InspectionMethodSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inspectionmethod.InspectionMethod InspectionMethod}
     * entity with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest}
     * method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code InspectionMethod}
     * @param inspectionMethodVersion
     *            Version Number of Inspection Method
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code InspectionMethod}
     * @param inspectionMethodPlant
     *            Plant for Inspection Method
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param inspectionMethod
     *            Inspection Method
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     */
    public InspectionMethodByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String inspectionMethod,
        final String inspectionMethodVersion,
        final String inspectionMethodPlant )
    {
        super(servicePath, entityCollection);
        this.key.put("InspectionMethod", inspectionMethod);
        this.key.put("InspectionMethodVersion", inspectionMethodVersion);
        this.key.put("InspectionMethodPlant", inspectionMethodPlant);
    }

    @Override
    @Nonnull
    protected Class<InspectionMethod> getEntityClass()
    {
        return InspectionMethod.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
