/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inspectionplan;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inspectionplan.selectable.InspPlanOperationSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inspectionplan.InspPlanOperation InspPlanOperation} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class InspPlanOperationByKeyFluentHelper
    extends
    FluentHelperByKey<InspPlanOperationByKeyFluentHelper, InspPlanOperation, InspPlanOperationSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.inspectionplan.InspPlanOperation InspPlanOperation}
     * entity with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest}
     * method on the fluent helper object.
     *
     * @param bOOOperationInternalID
     *            Number of the Task List Node
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code InspPlanOperation}
     * @param servicePath
     *            Service path to be used to fetch a single {@code InspPlanOperation}
     * @param inspectionPlan
     *            Group Counter
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param inspectionPlanInternalVersion
     *            Internal counter
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     * @param inspectionPlanGroup
     *            Key for Task List Group
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     * @param bOOOpInternalVersionCounter
     *            Internal counter
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     */
    public InspPlanOperationByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String inspectionPlanGroup,
        final String bOOOperationInternalID,
        final String inspectionPlanInternalVersion,
        final String bOOOpInternalVersionCounter,
        final String inspectionPlan )
    {
        super(servicePath, entityCollection);
        this.key.put("InspectionPlanGroup", inspectionPlanGroup);
        this.key.put("BOOOperationInternalID", bOOOperationInternalID);
        this.key.put("InspectionPlanInternalVersion", inspectionPlanInternalVersion);
        this.key.put("BOOOpInternalVersionCounter", bOOOpInternalVersionCounter);
        this.key.put("InspectionPlan", inspectionPlan);
    }

    @Override
    @Nonnull
    protected Class<InspPlanOperation> getEntityClass()
    {
        return InspPlanOperation.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
