/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.jointventurebilling;

import java.net.URI;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.CollectionValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>GetAttachmentLinks</b> OData function import.
 *
 */
public class GetAttachmentLinksFluentHelper
    extends
    CollectionValuedFluentHelperFunction<GetAttachmentLinksFluentHelper, AttachmentLink, List<AttachmentLink>>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>GetAttachmentLinks</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param companyCode
     *            Constraints: Not nullable, Maximum length: 4
     *            <p>
     *            Original parameter name from the Odata EDM: <b>CompanyCode</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param fiscalPeriod
     *            Constraints: Not nullable, Maximum length: 3
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FiscalPeriod</b>
     *            </p>
     * @param fiscalYear
     *            Constraints: Not nullable, Maximum length: 4
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FiscalYear</b>
     *            </p>
     * @param jointVenturePartner
     *            Constraints: Nullable, Maximum length: 10
     *            <p>
     *            Original parameter name from the Odata EDM: <b>JointVenturePartner</b>
     *            </p>
     */
    public GetAttachmentLinksFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String companyCode,
        @Nonnull final String fiscalYear,
        @Nonnull final String fiscalPeriod,
        @Nullable final String jointVenturePartner )
    {
        super(servicePath);
        values.put("CompanyCode", companyCode);
        values.put("FiscalYear", fiscalYear);
        values.put("FiscalPeriod", fiscalPeriod);
        if( jointVenturePartner != null ) {
            values.put("JointVenturePartner", jointVenturePartner);
        }
    }

    @Override
    @Nonnull
    protected Class<AttachmentLink> getEntityClass()
    {
        return AttachmentLink.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "GetAttachmentLinks";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpGet(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nonnull
    public List<AttachmentLink> executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeMultiple(destination);
    }

}
