/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.journalentryitem;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.journalentryitem.selectable.CostCenterSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.journalentryitem.CostCenter CostCenter} entity using key
 * fields. This fluent helper allows methods which modify the underlying query to be called before executing the query
 * itself.
 *
 */
public class CostCenterByKeyFluentHelper
    extends
    FluentHelperByKey<CostCenterByKeyFluentHelper, CostCenter, CostCenterSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.journalentryitem.CostCenter CostCenter} entity with
     * the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on
     * the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code CostCenter}
     * @param costCenter
     *            Cost Center
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code CostCenter}
     * @param controllingArea
     *            Controlling Area
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param validityEndDate
     *            Valid To Date
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     */
    public CostCenterByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String controllingArea,
        final String costCenter,
        final LocalDateTime validityEndDate )
    {
        super(servicePath, entityCollection);
        this.key.put("ControllingArea", controllingArea);
        this.key.put("CostCenter", costCenter);
        this.key.put("ValidityEndDate", validityEndDate);
    }

    @Override
    @Nonnull
    protected Class<CostCenter> getEntityClass()
    {
        return CostCenter.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
