/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.kanbancontrolcyclev2;

import java.math.BigDecimal;
import java.net.URI;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.CollectionValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>AddEventDrivenKanbanContainer</b> OData function import.
 *
 */
public class AddEventDrivenKanbanContainerFluentHelper
    extends
    CollectionValuedFluentHelperFunction<AddEventDrivenKanbanContainerFluentHelper, AddedKanbanContainer, List<AddedKanbanContainer>>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>AddEventDrivenKanbanContainer</b> OData function import
     * with the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param requestedDeliveryDateTime
     *            Time Stamp
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 7
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>RequestedDeliveryDateTime</b>
     *            </p>
     * @param numberOfLoadCarriers
     *            No.Load Carriers
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>NumberOfLoadCarriers</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param packingInstruction
     *            Packing instruction
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 20
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PackingInstruction</b>
     *            </p>
     * @param requestedQuantityInBaseUnit
     *            Requested Quantity
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 13, Scale: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>RequestedQuantityInBaseUnit</b>
     *            </p>
     * @param kanbanControlCycle
     *            Control Cycle
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 7
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>KanbanControlCycle</b>
     *            </p>
     */
    public AddEventDrivenKanbanContainerFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String kanbanControlCycle,
        @Nullable final BigDecimal requestedQuantityInBaseUnit,
        @Nullable final ZonedDateTime requestedDeliveryDateTime,
        @Nullable final String packingInstruction,
        @Nullable final Short numberOfLoadCarriers )
    {
        super(servicePath);
        values.put("KanbanControlCycle", kanbanControlCycle);
        if( requestedQuantityInBaseUnit != null ) {
            values.put("RequestedQuantityInBaseUnit", requestedQuantityInBaseUnit);
        }
        if( requestedDeliveryDateTime != null ) {
            values.put("RequestedDeliveryDateTime", requestedDeliveryDateTime);
        }
        if( packingInstruction != null ) {
            values.put("PackingInstruction", packingInstruction);
        }
        if( numberOfLoadCarriers != null ) {
            values.put("NumberOfLoadCarriers", numberOfLoadCarriers);
        }
    }

    @Override
    @Nonnull
    protected Class<AddedKanbanContainer> getEntityClass()
    {
        return AddedKanbanContainer.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "AddEventDrivenKanbanContainer";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nonnull
    public List<AddedKanbanContainer> executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeMultiple(destination);
    }

}
