/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.maintenancebillofmaterial;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>CreateVariantBOM</b> OData function import.
 *
 */
public class CreateVariantBOMFluentHelper
    extends
    SingleValuedFluentHelperFunction<CreateVariantBOMFluentHelper, DummyFunctionImportResult, DummyFunctionImportResult>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>CreateVariantBOM</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param billOfMaterialCategory
     *            BOM category
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialCategory</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param plant
     *            Plant
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Plant</b>
     *            </p>
     * @param headerValidityStartDate
     *            Valid From
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>HeaderValidityStartDate</b>
     *            </p>
     * @param engineeringChangeDocument
     *            Change Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>EngineeringChangeDocument</b>
     *            </p>
     * @param technicalObject
     *            Technical Object
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>TechnicalObject</b>
     *            </p>
     * @param billOfMaterialVariantUsage
     *            BOM Usage
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialVariantUsage</b>
     *            </p>
     * @param referenceObjectID
     *            Owner object ID
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 70
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ReferenceObjectID</b>
     *            </p>
     */
    public CreateVariantBOMFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String technicalObject,
        @Nonnull final String billOfMaterialVariantUsage,
        @Nonnull final String plant,
        @Nonnull final String billOfMaterialCategory,
        @Nonnull final String referenceObjectID,
        @Nullable final LocalDateTime headerValidityStartDate,
        @Nullable final String engineeringChangeDocument )
    {
        super(servicePath);
        values.put("TechnicalObject", technicalObject);
        values.put("BillOfMaterialVariantUsage", billOfMaterialVariantUsage);
        values.put("Plant", plant);
        values.put("BillOfMaterialCategory", billOfMaterialCategory);
        values.put("ReferenceObjectID", referenceObjectID);
        if( headerValidityStartDate != null ) {
            values.put("HeaderValidityStartDate", headerValidityStartDate);
        }
        if( engineeringChangeDocument != null ) {
            values.put("EngineeringChangeDocument", engineeringChangeDocument);
        }
    }

    @Override
    @Nonnull
    protected Class<DummyFunctionImportResult> getEntityClass()
    {
        return DummyFunctionImportResult.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "CreateVariantBOM";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public DummyFunctionImportResult executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
