/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.maintenanceorder;

import java.net.URI;
import java.time.ZonedDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>SetMaintOrdToMainWorkComplete</b> OData function import.
 *
 */
public class SetMaintOrdToMainWorkCompleteFluentHelper
    extends
    SingleValuedFluentHelperFunction<SetMaintOrdToMainWorkCompleteFluentHelper, DummyFunctionImportResult, DummyFunctionImportResult>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>SetMaintOrdToMainWorkComplete</b> OData function import
     * with the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param maintenanceOrder
     *            Order
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaintenanceOrder</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     */
    public SetMaintOrdToMainWorkCompleteFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String maintenanceOrder )
    {
        super(servicePath);
        values.put("MaintenanceOrder", maintenanceOrder);
    }

    /**
     * Creates a fluent helper object that will execute the <b>SetMaintOrdToMainWorkComplete</b> OData function import
     * with the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param maintenanceOrder
     *            Order
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaintenanceOrder</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param mainWrkCmpltdDateTimeIsUsed
     *            Set as Reference Date and Time
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MainWrkCmpltdDateTimeIsUsed</b>
     *            </p>
     * @param mainWorkCompletedDateTime
     *            Main Work Completion Date and Time
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MainWorkCompletedDateTime</b>
     *            </p>
     */
    public SetMaintOrdToMainWorkCompleteFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String maintenanceOrder,
        @Nullable final ZonedDateTime mainWorkCompletedDateTime,
        @Nullable final Boolean mainWrkCmpltdDateTimeIsUsed )
    {
        super(servicePath);
        values.put("MaintenanceOrder", maintenanceOrder);
        if( mainWorkCompletedDateTime != null ) {
            values.put("MainWorkCompletedDateTime", mainWorkCompletedDateTime);
        }
        if( mainWrkCmpltdDateTimeIsUsed != null ) {
            values.put("MainWrkCmpltdDateTimeIsUsed", mainWrkCmpltdDateTimeIsUsed);
        }
    }

    @Override
    @Nonnull
    protected Class<DummyFunctionImportResult> getEntityClass()
    {
        return DummyFunctionImportResult.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "SetMaintOrdToMainWorkComplete";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public DummyFunctionImportResult executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
