/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.maintenanceplan;

import java.net.URI;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>StartMaintPlnSchedule</b> OData function import.
 *
 */
public class StartMaintPlnScheduleFluentHelper
    extends
    SingleValuedFluentHelperFunction<StartMaintPlnScheduleFluentHelper, MaintenancePlan, MaintenancePlan>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>StartMaintPlnSchedule</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param maintenancePlan
     *            Maintenance Plan
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaintenancePlan</b>
     *            </p>
     * @param schedulingStartDate
     *            Start of cycle
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SchedulingStartDate</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param schedulingStartTime
     *            Start Time
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SchedulingStartTime</b>
     *            </p>
     * @param maintPlanStartCntrReadingValue
     *            StartCounterReading
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 22
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaintPlanStartCntrReadingValue</b>
     *            </p>
     * @param maintPlnSchedgCallObjUpToDte
     *            Current Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaintPlnSchedgCallObjUpToDte</b>
     *            </p>
     */
    public StartMaintPlnScheduleFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String maintenancePlan,
        @Nullable final LocalDateTime schedulingStartDate,
        @Nullable final LocalTime schedulingStartTime,
        @Nullable final String maintPlanStartCntrReadingValue,
        @Nullable final LocalDateTime maintPlnSchedgCallObjUpToDte )
    {
        super(servicePath);
        values.put("MaintenancePlan", maintenancePlan);
        if( schedulingStartDate != null ) {
            values.put("SchedulingStartDate", schedulingStartDate);
        }
        if( schedulingStartTime != null ) {
            values.put("SchedulingStartTime", schedulingStartTime);
        }
        if( maintPlanStartCntrReadingValue != null ) {
            values.put("MaintPlanStartCntrReadingValue", maintPlanStartCntrReadingValue);
        }
        if( maintPlnSchedgCallObjUpToDte != null ) {
            values.put("MaintPlnSchedgCallObjUpToDte", maintPlnSchedgCallObjUpToDte);
        }
    }

    @Override
    @Nonnull
    protected Class<MaintenancePlan> getEntityClass()
    {
        return MaintenancePlan.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "StartMaintPlnSchedule";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public MaintenancePlan executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
