/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.masterinspectioncharacteristic;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.masterinspectioncharacteristic.selectable.InspectionSpecificationSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.masterinspectioncharacteristic.InspectionSpecification
 * InspectionSpecification} entity using key fields. This fluent helper allows methods which modify the underlying query
 * to be called before executing the query itself.
 *
 */
public class InspectionSpecificationByKeyFluentHelper
    extends
    FluentHelperByKey<InspectionSpecificationByKeyFluentHelper, InspectionSpecification, InspectionSpecificationSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.masterinspectioncharacteristic.InspectionSpecification
     * InspectionSpecification} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code InspectionSpecification}
     * @param inspectionSpecificationPlant
     *            Plant for Master Inspection Characteristic
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param inspectionSpecification
     *            Master Inspection Characteristic
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code InspectionSpecification}
     * @param inspectionSpecificationVersion
     *            Version Number of Master Inspection Characteristic
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     */
    public InspectionSpecificationByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String inspectionSpecification,
        final String inspectionSpecificationVersion,
        final String inspectionSpecificationPlant )
    {
        super(servicePath, entityCollection);
        this.key.put("InspectionSpecification", inspectionSpecification);
        this.key.put("InspectionSpecificationVersion", inspectionSpecificationVersion);
        this.key.put("InspectionSpecificationPlant", inspectionSpecificationPlant);
    }

    @Override
    @Nonnull
    protected Class<InspectionSpecification> getEntityClass()
    {
        return InspectionSpecification.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
