/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialdocument;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>CancelItem</b> OData function import.
 *
 */
public class CancelItemFluentHelper
    extends
    SingleValuedFluentHelperFunction<CancelItemFluentHelper, MaterialDocumentItem, MaterialDocumentItem>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>CancelItem</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param materialDocument
     *            Material Document
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaterialDocument</b>
     *            </p>
     * @param materialDocumentItem
     *            Material Doc.Item
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaterialDocumentItem</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param materialDocumentYear
     *            Material Doc. Year
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaterialDocumentYear</b>
     *            </p>
     * @param postingDate
     *            Posting Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PostingDate</b>
     *            </p>
     */
    public CancelItemFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String materialDocumentYear,
        @Nonnull final String materialDocument,
        @Nonnull final String materialDocumentItem,
        @Nullable final LocalDateTime postingDate )
    {
        super(servicePath);
        values.put("MaterialDocumentYear", materialDocumentYear);
        values.put("MaterialDocument", materialDocument);
        values.put("MaterialDocumentItem", materialDocumentItem);
        if( postingDate != null ) {
            values.put("PostingDate", postingDate);
        }
    }

    @Override
    @Nonnull
    protected Class<MaterialDocumentItem> getEntityClass()
    {
        return MaterialDocumentItem.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "CancelItem";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public MaterialDocumentItem executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
