/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialdocument;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialdocument.selectable.MaterialDocumentItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialdocument.MaterialDocumentItem
 * MaterialDocumentItem} entity using key fields. This fluent helper allows methods which modify the underlying query to
 * be called before executing the query itself.
 *
 */
public class MaterialDocumentItemByKeyFluentHelper
    extends
    FluentHelperByKey<MaterialDocumentItemByKeyFluentHelper, MaterialDocumentItem, MaterialDocumentItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialdocument.MaterialDocumentItem
     * MaterialDocumentItem} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param materialDocument
     *            Number of Material Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code MaterialDocumentItem}
     * @param materialDocumentItem
     *            Material Document Item
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code MaterialDocumentItem}
     * @param materialDocumentYear
     *            Material Document Year
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public MaterialDocumentItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String materialDocumentYear,
        final String materialDocument,
        final String materialDocumentItem )
    {
        super(servicePath, entityCollection);
        this.key.put("MaterialDocumentYear", materialDocumentYear);
        this.key.put("MaterialDocument", materialDocument);
        this.key.put("MaterialDocumentItem", materialDocumentItem);
    }

    @Override
    @Nonnull
    protected Class<MaterialDocumentItem> getEntityClass()
    {
        return MaterialDocumentItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
