/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialplanningdata;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialplanningdata.selectable.MRPMaterialSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialplanningdata.MRPMaterial MRPMaterial} entity using
 * key fields. This fluent helper allows methods which modify the underlying query to be called before executing the
 * query itself.
 *
 */
public class MRPMaterialByKeyFluentHelper
    extends
    FluentHelperByKey<MRPMaterialByKeyFluentHelper, MRPMaterial, MRPMaterialSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialplanningdata.MRPMaterial MRPMaterial} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code MRPMaterial}
     * @param mRPArea
     *            MRP Area
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param material
     *            Material Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code MRPMaterial}
     * @param mRPPlant
     *            Plant
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public MRPMaterialByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String material,
        final String mRPPlant,
        final String mRPArea )
    {
        super(servicePath, entityCollection);
        this.key.put("Material", material);
        this.key.put("MRPPlant", mRPPlant);
        this.key.put("MRPArea", mRPArea);
    }

    @Override
    @Nonnull
    protected Class<MRPMaterial> getEntityClass()
    {
        return MRPMaterial.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
