/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialprice;

import java.math.BigDecimal;
import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>InventoryPriceChange</b> OData function import.
 *
 */
public class InventoryPriceChangeFluentHelper
    extends
    SingleValuedFluentHelperFunction<InventoryPriceChangeFluentHelper, PriceChangeDocument, PriceChangeDocument>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>InventoryPriceChange</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param valuationArea
     *            Valuation Area
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ValuationArea</b>
     *            </p>
     * @param material
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Material</b>
     *            </p>
     * @param materialPriceUnitQty
     *            Price unit
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Precision: 5, Scale: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaterialPriceUnitQty</b>
     *            </p>
     * @param inventoryValuationType
     *            Valuation Type
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>InventoryValuationType</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param currency
     *            Currency
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 5
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Currency</b>
     *            </p>
     * @param postingDate
     *            Posting Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PostingDate</b>
     *            </p>
     * @param currencyRole
     *            Crcy type/val.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>CurrencyRole</b>
     *            </p>
     * @param inventoryPrice
     *            Price
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Precision: 4, Scale: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>InventoryPrice</b>
     *            </p>
     */
    public InventoryPriceChangeFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String currency,
        @Nonnull final String currencyRole,
        @Nonnull final String material,
        @Nonnull final BigDecimal inventoryPrice,
        @Nonnull final LocalDateTime postingDate,
        @Nonnull final BigDecimal materialPriceUnitQty,
        @Nonnull final String valuationArea,
        @Nonnull final String inventoryValuationType )
    {
        super(servicePath);
        values.put("Currency", currency);
        values.put("CurrencyRole", currencyRole);
        values.put("Material", material);
        values.put("InventoryPrice", inventoryPrice);
        values.put("PostingDate", postingDate);
        values.put("MaterialPriceUnitQty", materialPriceUnitQty);
        values.put("ValuationArea", valuationArea);
        values.put("InventoryValuationType", inventoryValuationType);
    }

    @Override
    @Nonnull
    protected Class<PriceChangeDocument> getEntityClass()
    {
        return PriceChangeDocument.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "InventoryPriceChange";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public PriceChangeDocument executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
