/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialstockdisplay;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialstockdisplay.selectable.MatlStkInAcctModSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialstockdisplay.MatlStkInAcctMod MatlStkInAcctMod}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class MatlStkInAcctModByKeyFluentHelper
    extends
    FluentHelperByKey<MatlStkInAcctModByKeyFluentHelper, MatlStkInAcctMod, MatlStkInAcctModSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.materialstockdisplay.MatlStkInAcctMod
     * MatlStkInAcctMod} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param wBSElementInternalID
     *            Valuated Sales Order Stock WBS Element
     *            <p>
     *            Constraints: Not nullable, Maximum length: 24
     *            </p>
     * @param batch
     *            Batch Number (Stock Identifier)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param storageLocation
     *            Storage Location
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param sDDocument
     *            Sales order number of valuated sales order stock
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code MatlStkInAcctMod}
     * @param material
     *            Material in Respect of Which Stock is Managed
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     * @param inventorySpecialStockType
     *            Special Stock Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code MatlStkInAcctMod}
     * @param plant
     *            Plant
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param supplier
     *            Supplier for Special Stock
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param sDDocumentItem
     *            Sales Order Item of Valuated Sales Order Stock
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param inventoryStockType
     *            Stock Type of Goods Movement (Stock Identifier)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param customer
     *            Customer for Special Stock
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public MatlStkInAcctModByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String material,
        final String plant,
        final String storageLocation,
        final String batch,
        final String supplier,
        final String customer,
        final String wBSElementInternalID,
        final String sDDocument,
        final String sDDocumentItem,
        final String inventorySpecialStockType,
        final String inventoryStockType )
    {
        super(servicePath, entityCollection);
        this.key.put("Material", material);
        this.key.put("Plant", plant);
        this.key.put("StorageLocation", storageLocation);
        this.key.put("Batch", batch);
        this.key.put("Supplier", supplier);
        this.key.put("Customer", customer);
        this.key.put("WBSElementInternalID", wBSElementInternalID);
        this.key.put("SDDocument", sDDocument);
        this.key.put("SDDocumentItem", sDDocumentItem);
        this.key.put("InventorySpecialStockType", inventorySpecialStockType);
        this.key.put("InventoryStockType", inventoryStockType);
    }

    @Override
    @Nonnull
    protected Class<MatlStkInAcctMod> getEntityClass()
    {
        return MatlStkInAcctMod.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
