/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.outbounddeliveryv2;

import java.math.BigDecimal;
import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>CreateBatchSplitItem</b> OData function import.
 *
 */
public class CreateBatchSplitItemFluentHelper
    extends
    SingleValuedFluentHelperFunction<CreateBatchSplitItemFluentHelper, CreatedDeliveryItem, CreatedDeliveryItem>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>CreateBatchSplitItem</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param deliveryQuantityUnit
     *            Sales Unit
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DeliveryQuantityUnit</b>
     *            </p>
     * @param deliveryDocument
     *            Delivery
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DeliveryDocument</b>
     *            </p>
     * @param pickQuantityInSalesUOM
     *            Pick quantity
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 3, Scale: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PickQuantityInSalesUOM</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param batch
     *            Batch
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Batch</b>
     *            </p>
     * @param actualDeliveryQuantity
     *            Delivery Qty
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Precision: 3, Scale: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ActualDeliveryQuantity</b>
     *            </p>
     * @param deliveryDocumentItem
     *            Item
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DeliveryDocumentItem</b>
     *            </p>
     */
    public CreateBatchSplitItemFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String deliveryQuantityUnit,
        @Nonnull final BigDecimal actualDeliveryQuantity,
        @Nonnull final String deliveryDocumentItem,
        @Nonnull final String deliveryDocument,
        @Nonnull final String batch,
        @Nullable final BigDecimal pickQuantityInSalesUOM )
    {
        super(servicePath);
        values.put("DeliveryQuantityUnit", deliveryQuantityUnit);
        values.put("ActualDeliveryQuantity", actualDeliveryQuantity);
        values.put("DeliveryDocumentItem", deliveryDocumentItem);
        values.put("DeliveryDocument", deliveryDocument);
        values.put("Batch", batch);
        if( pickQuantityInSalesUOM != null ) {
            values.put("PickQuantityInSalesUOM", pickQuantityInSalesUOM);
        }
    }

    @Override
    @Nonnull
    protected Class<CreatedDeliveryItem> getEntityClass()
    {
        return CreatedDeliveryItem.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "CreateBatchSplitItem";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public CreatedDeliveryItem executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
