/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.physicalinventorydocument;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>InitiateRecountOnItem</b> OData function import.
 *
 */
public class InitiateRecountOnItemFluentHelper
    extends
    SingleValuedFluentHelperFunction<InitiateRecountOnItemFluentHelper, PhysInventoryDocItem, PhysInventoryDocItem>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>InitiateRecountOnItem</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param documentDate
     *            Document Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DocumentDate</b>
     *            </p>
     * @param physicalInventoryDocumentItem
     *            Item
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysicalInventoryDocumentItem</b>
     *            </p>
     * @param physInvtryDocHasQtySnapshot
     *            Freeze Book Invtry
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysInvtryDocHasQtySnapshot</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param physInventoryPlannedCountDate
     *            Planned Count Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysInventoryPlannedCountDate</b>
     *            </p>
     * @param postingIsBlockedForPhysInvtry
     *            Phys. Inv. Block
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PostingIsBlockedForPhysInvtry</b>
     *            </p>
     * @param fiscalYear
     *            Fiscal Year
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FiscalYear</b>
     *            </p>
     * @param physicalInventoryNumber
     *            Phys. inventory no.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 16
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysicalInventoryNumber</b>
     *            </p>
     * @param physicalInventoryDocument
     *            Phys. Inventory Doc.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysicalInventoryDocument</b>
     *            </p>
     * @param physInventoryReferenceNumber
     *            Phys. inventory ref.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 16
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysInventoryReferenceNumber</b>
     *            </p>
     * @param physicalInventoryDocumentDesc
     *            Description
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysicalInventoryDocumentDesc</b>
     *            </p>
     */
    public InitiateRecountOnItemFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String physicalInventoryDocument,
        @Nonnull final String fiscalYear,
        @Nonnull final String physicalInventoryDocumentItem,
        @Nullable final LocalDateTime physInventoryPlannedCountDate,
        @Nullable final LocalDateTime documentDate,
        @Nullable final String physicalInventoryNumber,
        @Nullable final String physInventoryReferenceNumber,
        @Nullable final String physicalInventoryDocumentDesc,
        @Nullable final Boolean physInvtryDocHasQtySnapshot,
        @Nullable final Boolean postingIsBlockedForPhysInvtry )
    {
        super(servicePath);
        values.put("PhysicalInventoryDocument", physicalInventoryDocument);
        values.put("FiscalYear", fiscalYear);
        values.put("PhysicalInventoryDocumentItem", physicalInventoryDocumentItem);
        if( physInventoryPlannedCountDate != null ) {
            values.put("PhysInventoryPlannedCountDate", physInventoryPlannedCountDate);
        }
        if( documentDate != null ) {
            values.put("DocumentDate", documentDate);
        }
        if( physicalInventoryNumber != null ) {
            values.put("PhysicalInventoryNumber", physicalInventoryNumber);
        }
        if( physInventoryReferenceNumber != null ) {
            values.put("PhysInventoryReferenceNumber", physInventoryReferenceNumber);
        }
        if( physicalInventoryDocumentDesc != null ) {
            values.put("PhysicalInventoryDocumentDesc", physicalInventoryDocumentDesc);
        }
        if( physInvtryDocHasQtySnapshot != null ) {
            values.put("PhysInvtryDocHasQtySnapshot", physInvtryDocHasQtySnapshot);
        }
        if( postingIsBlockedForPhysInvtry != null ) {
            values.put("PostingIsBlockedForPhysInvtry", postingIsBlockedForPhysInvtry);
        }
    }

    @Override
    @Nonnull
    protected Class<PhysInventoryDocItem> getEntityClass()
    {
        return PhysInventoryDocItem.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "InitiateRecountOnItem";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public PhysInventoryDocItem executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
