/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.physicalinventorydocument;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.physicalinventorydocument.selectable.PhysInventoryDocItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.physicalinventorydocument.PhysInventoryDocItem
 * PhysInventoryDocItem} entity using key fields. This fluent helper allows methods which modify the underlying query to
 * be called before executing the query itself.
 *
 */
public class PhysInventoryDocItemByKeyFluentHelper
    extends
    FluentHelperByKey<PhysInventoryDocItemByKeyFluentHelper, PhysInventoryDocItem, PhysInventoryDocItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.physicalinventorydocument.PhysInventoryDocItem
     * PhysInventoryDocItem} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code PhysInventoryDocItem}
     * @param physicalInventoryDocumentItem
     *            Physical Inventory Item Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code PhysInventoryDocItem}
     * @param fiscalYear
     *            Fiscal Year
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param physicalInventoryDocument
     *            Physical Inventory Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public PhysInventoryDocItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String fiscalYear,
        final String physicalInventoryDocument,
        final String physicalInventoryDocumentItem )
    {
        super(servicePath, entityCollection);
        this.key.put("FiscalYear", fiscalYear);
        this.key.put("PhysicalInventoryDocument", physicalInventoryDocument);
        this.key.put("PhysicalInventoryDocumentItem", physicalInventoryDocumentItem);
    }

    @Override
    @Nonnull
    protected Class<PhysInventoryDocItem> getEntityClass()
    {
        return PhysInventoryDocItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
