/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.physicalinventorydocument;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>PostDifferencesOnItem</b> OData function import.
 *
 */
public class PostDifferencesOnItemFluentHelper
    extends
    SingleValuedFluentHelperFunction<PostDifferencesOnItemFluentHelper, PhysInventoryDocItem, PhysInventoryDocItem>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>PostDifferencesOnItem</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param physicalInventoryDocumentItem
     *            Item
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysicalInventoryDocumentItem</b>
     *            </p>
     * @param material
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Material</b>
     *            </p>
     * @param reasonForPhysInvtryDifference
     *            Reason f. Diff.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ReasonForPhysInvtryDifference</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param batch
     *            Batch
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Batch</b>
     *            </p>
     * @param postingDate
     *            Posting Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PostingDate</b>
     *            </p>
     * @param fiscalYear
     *            Fiscal Year
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FiscalYear</b>
     *            </p>
     * @param physicalInventoryDocument
     *            Phys. Inventory Doc.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PhysicalInventoryDocument</b>
     *            </p>
     */
    public PostDifferencesOnItemFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String physicalInventoryDocumentItem,
        @Nonnull final String material,
        @Nonnull final String physicalInventoryDocument,
        @Nonnull final String fiscalYear,
        @Nullable final String batch,
        @Nullable final String reasonForPhysInvtryDifference,
        @Nullable final LocalDateTime postingDate )
    {
        super(servicePath);
        values.put("PhysicalInventoryDocumentItem", physicalInventoryDocumentItem);
        values.put("Material", material);
        values.put("PhysicalInventoryDocument", physicalInventoryDocument);
        values.put("FiscalYear", fiscalYear);
        if( batch != null ) {
            values.put("Batch", batch);
        }
        if( reasonForPhysInvtryDifference != null ) {
            values.put("ReasonForPhysInvtryDifference", reasonForPhysInvtryDifference);
        }
        if( postingDate != null ) {
            values.put("PostingDate", postingDate);
        }
    }

    @Override
    @Nonnull
    protected Class<PhysInventoryDocItem> getEntityClass()
    {
        return PhysInventoryDocItem.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "PostDifferencesOnItem";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public PhysInventoryDocItem executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
